<?php
/**
 * Plugin Name: SnapPress Connect
 * Plugin URI: https://snappress.app
 * Description: Generate a QR code to easily register your WordPress site with the SnapPress iOS app.
 * Version: 1.1.0
 * Author: 37design
 * Author URI: https://37design.co.jp
 * License: GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: snappress-connect
 * Domain Path: /languages
 * Requires at least: 5.6
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class SnapPress_Connect {

	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_menu_page' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
	}

	public function add_menu_page() {
		add_management_page(
			esc_html__( 'SnapPress Connect', 'snappress-connect' ),
			esc_html__( 'SnapPress Connect', 'snappress-connect' ),
			'read',
			'snappress-connect',
			array( $this, 'render_page' )
		);
	}

	public function enqueue_assets( $hook ) {
		if ( 'tools_page_snappress-connect' !== $hook ) {
			return;
		}

		wp_enqueue_script(
			'qrcodejs',
			plugin_dir_url( __FILE__ ) . 'js/qrcode.min.js',
			array(),
			'1.0.0',
			true
		);

		wp_enqueue_script(
			'snappress-connect',
			plugin_dir_url( __FILE__ ) . 'js/snappress-connect.js',
			array( 'qrcodejs', 'wp-i18n' ),
			'1.1.0',
			true
		);

		wp_set_script_translations( 'snappress-connect', 'snappress-connect', plugin_dir_path( __FILE__ ) . 'languages' );

		$current_user = wp_get_current_user();

		wp_localize_script( 'snappress-connect', 'snappressConnect', array(
			'siteUrl'   => home_url(),
			'username'  => $current_user->user_login,
			'restUrl'   => esc_url_raw( rest_url( 'wp/v2/users/me/application-passwords' ) ),
			'restNonce' => wp_create_nonce( 'wp_rest' ),
		) );
	}

	public function render_page() {
		$current_user = wp_get_current_user();
		$site_url     = home_url();
		$username     = esc_attr( $current_user->user_login );
		?>
		<div class="wrap">
			<h1><?php echo esc_html__( 'SnapPress Connect', 'snappress-connect' ); ?></h1>
			<p><?php echo esc_html__( 'Generate a QR code to register this site with the SnapPress app.', 'snappress-connect' ); ?></p>

			<div style="margin: 16px 0; padding: 20px 24px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 8px; display: flex; align-items: center; gap: 20px; color: #fff;">
				<div style="flex-shrink: 0;">
					<svg width="48" height="48" viewBox="0 0 48 48" fill="none" xmlns="http://www.w3.org/2000/svg">
						<rect width="48" height="48" rx="10" fill="rgba(255,255,255,0.2)"/>
						<path d="M24 12L28 20H20L24 12Z" fill="white" opacity="0.9"/>
						<rect x="14" y="22" width="20" height="14" rx="2" fill="white" opacity="0.9"/>
						<circle cx="24" cy="29" r="4" fill="#764ba2"/>
					</svg>
				</div>
				<div style="flex: 1;">
					<div style="font-size: 16px; font-weight: 700; margin-bottom: 4px;">
						<?php echo esc_html__( 'Need the SnapPress app?', 'snappress-connect' ); ?>
					</div>
					<div style="font-size: 13px; opacity: 0.9; margin-bottom: 10px;">
						<?php echo esc_html__( 'Batch upload photos from your iPhone to WordPress.', 'snappress-connect' ); ?>
					</div>
					<a href="https://snappress.app" target="_blank" rel="noopener noreferrer" style="display: inline-block; background: #fff; color: #333; text-decoration: none; padding: 8px 20px; border-radius: 6px; font-size: 13px; font-weight: 600;">
						<?php echo esc_html__( 'Download on the App Store', 'snappress-connect' ); ?> &rarr;
					</a>
				</div>
			</div>

			<table class="form-table">
				<tr>
					<th scope="row"><?php echo esc_html__( 'Site URL', 'snappress-connect' ); ?></th>
					<td><code><?php echo esc_html( $site_url ); ?></code></td>
				</tr>
				<tr>
					<th scope="row"><?php echo esc_html__( 'Username', 'snappress-connect' ); ?></th>
					<td><code><?php echo esc_html( $username ); ?></code></td>
				</tr>
				<tr>
					<th scope="row">Application Password</th>
					<td>
						<p>
							<button type="button" id="snappress-oneclick" class="button button-primary button-hero">
								<?php echo esc_html__( 'One-Tap QR Code Generation', 'snappress-connect' ); ?>
							</button>
						</p>
						<p class="description" style="margin-top: 8px;">
							<?php echo esc_html__( 'Automatically issues an Application Password and displays a QR code.', 'snappress-connect' ); ?>
						</p>

						<details style="margin-top: 16px;">
							<summary style="cursor: pointer; color: #2271b1;"><?php echo esc_html__( 'Use Existing Application Password', 'snappress-connect' ); ?></summary>
							<div style="margin-top: 8px;">
								<input type="password" id="snappress-app-password" class="regular-text" placeholder="XXXX XXXX XXXX XXXX" autocomplete="off" />
								<p class="description">
									<a href="<?php echo esc_url( admin_url( 'profile.php#application-passwords-section' ) ); ?>"><?php echo esc_html__( 'Users → Profile → Application Passwords', 'snappress-connect' ); ?></a>
								</p>
								<p style="margin-top: 8px;">
									<button type="button" id="snappress-generate" class="button" disabled><?php echo esc_html__( 'Generate QR Code', 'snappress-connect' ); ?></button>
								</p>
							</div>
						</details>
					</td>
				</tr>
			</table>

			<div id="snappress-status" style="margin-top: 16px;"></div>
			<div id="snappress-qrcode" style="margin-top: 20px;"></div>
			<p id="snappress-notice" style="display:none; margin-top: 10px; color: #d63638; font-weight: bold;">
				<?php echo esc_html__( 'This QR code contains login credentials. Please close this page after scanning.', 'snappress-connect' ); ?>
			</p>
			<p>
				<button type="button" id="snappress-clear" class="button" style="display:none;"><?php echo esc_html__( 'Clear', 'snappress-connect' ); ?></button>
			</p>
		</div>
		<?php
	}
}

new SnapPress_Connect();
