/**
 * SnapPress Connect - Admin Page Script
 *
 * @package SnapPress_Connect
 * @since 1.1.0
 */

/* global snappressConnect, QRCode, wp */

(function() {
	'use strict';

	var __ = wp.i18n.__;

	var siteUrl   = snappressConnect.siteUrl;
	var username  = snappressConnect.username;
	var restUrl   = snappressConnect.restUrl;
	var restNonce = snappressConnect.restNonce;

	var oneclickBtn   = document.getElementById( 'snappress-oneclick' );
	var passwordInput = document.getElementById( 'snappress-app-password' );
	var generateBtn   = document.getElementById( 'snappress-generate' );
	var clearBtn      = document.getElementById( 'snappress-clear' );
	var qrContainer   = document.getElementById( 'snappress-qrcode' );
	var notice        = document.getElementById( 'snappress-notice' );
	var status        = document.getElementById( 'snappress-status' );

	passwordInput.addEventListener( 'input', function() {
		generateBtn.disabled = ! this.value.trim();
	} );

	// One-Tap: Automatically issue Application Password then generate QR
	oneclickBtn.addEventListener( 'click', function() {
		oneclickBtn.disabled    = true;
		oneclickBtn.textContent = __( 'Issuing...', 'snappress-connect' );
		status.innerHTML        = '';

		fetch( restUrl, {
			method: 'POST',
			headers: {
				'Content-Type': 'application/json',
				'X-WP-Nonce': restNonce
			},
			body: JSON.stringify( { name: 'SnapPress' } )
		} )
		.then( function( res ) {
			if ( ! res.ok ) {
				return res.json().then( function( err ) {
					throw new Error( err.message || __( 'Failed to issue password', 'snappress-connect' ) );
				} );
			}
			return res.json();
		} )
		.then( function( data ) {
			showQR( data.password );
			oneclickBtn.style.display = 'none';
		} )
		.catch( function( err ) {
			status.innerHTML        = '<div class="notice notice-error inline"><p>' + escapeHtml( err.message ) + '</p></div>';
			oneclickBtn.disabled    = false;
			oneclickBtn.textContent = __( 'One-Tap QR Code Generation', 'snappress-connect' );
		} );
	} );

	// Manual input: Generate QR
	generateBtn.addEventListener( 'click', function() {
		var password = passwordInput.value.trim();
		if ( ! password ) {
			return;
		}
		showQR( password );
	} );

	function showQR( password ) {
		var data = JSON.stringify( {
			snappress: 1,
			url: siteUrl,
			user: username,
			pass: password
		} );

		qrContainer.innerHTML = '';
		new QRCode( qrContainer, {
			text: data,
			width: 256,
			height: 256,
			colorDark: '#000000',
			colorLight: '#ffffff',
			correctLevel: QRCode.CorrectLevel.M
		} );

		notice.style.display      = 'block';
		clearBtn.style.display    = '';
		generateBtn.style.display = 'none';
		status.innerHTML          = '';
	}

	clearBtn.addEventListener( 'click', function() {
		qrContainer.innerHTML     = '';
		notice.style.display      = 'none';
		clearBtn.style.display    = 'none';
		generateBtn.style.display = '';
		generateBtn.disabled      = true;
		passwordInput.value       = '';
		oneclickBtn.style.display = '';
		oneclickBtn.disabled      = false;
		oneclickBtn.textContent   = __( 'One-Tap QR Code Generation', 'snappress-connect' );
		status.innerHTML          = '';
	} );

	function escapeHtml( str ) {
		var div = document.createElement( 'div' );
		div.appendChild( document.createTextNode( str ) );
		return div.innerHTML;
	}
})();
